/*
 * Custom half-assed mostly ground-up rewrite of Dapple using Allegro
 * for the special purpose of running Star Emulator.
 *   (IE IT EMULATES THE APPLE /// GRAPHICS MODES.)
 * This is not a general purpose Apple ][ or Apple /// emu and will not work
 * as one.  It does not support most of the Apple ///'s functionality and it
 * most certainly cannot run SOS.  It's just enough to get the essentials up.
 *   New code (C) 2012 Steve Nickolas.  Other code has its own copyrights.
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <allegro.h>
#include "M6502.h"

#define DAPLUS_VERSION "Daplus 3 version 0.00"
#define DAPLUS_DATE    "2012"

static unsigned char mem[65536];
static unsigned char undermem[32768][5];
static unsigned char rom[4096]; /* only used by the /// */
unsigned char starfont[1024], sysfont[1024], *myfont;
M6502 CPU;
int corpse;
int real3, isrom;
int hideio, lastkey, fstack, cpuspeed;
int hasstarfont, hassysfont;
static int ctdif;
unsigned long ClockTick;

byte tty_color, tty_dblres, tty_page2, tty_grx;

static unsigned long curbuf[640][480];

int FastMode = 0;
/* 0:correctly emulate drive rotation speed, !0:faster, but
   the drive performs one rotation in 130.5ms (instead of
   200ms). Turn fast mode off so ProDOS formatting and some
   copy-protection schemes can work. */

/* applemu incorrectly uses 0x1a00 length tracks. A real
   Apple disk drive spinning at ~300RPM, or 1 rev every
   200ms and writing a byte every 32us has a capacity
   of 6250 bytes/track not 6656. Setting TrackBufLen to
   0x1a00 allows my disk emulation to read applemu disks,
   but with FastMode off the drive 'spins' a little too
   slow, and the extra bytes that are available on every
   track can confuse formatting programs. The ProDOS FILER
   is one of these.  You MUST set FastMode to 0 and
   TrackBufLen to 6250 for FILER to format. */
int TrackBufLen = 6250;
long LastIO = 0;
long Diff, LeftOverCycles;

/*
 * Star Emulator manipulates FFDF to lock and unlock the firmware area.
 * We need to emulate this even if we're not implementing the full ///.
 *
 *    if Off                if On
 * 80 cpu run 2 MHz*        cpu run 1 MHz
 * 40 ram at c000.c4ff      mapped i/o and slot roms at c000.c4ff
 * 20 disable video         enable video
 * 10 disable reset         enable reset
 * 08 unlock the rom area   lock the rom area
 * 04 float the stack**     lock the stack
 * 02 bank the rom***       ram at f000.ffff
 * 01 hide the rom***       reveal the rom***
 *
 * * disk drive support depends on a 1 MHz cpu clock, so leave it ON!
 * ** variable location zero page and stack.  we have no need for it.
 * *** apple3.rom - unused.
 *
 * We use the following settings:
 *    F0 - 11110000 - unprotects the ROM (floats the stack but we
 *                    never use the stack while setting up star emu)
 *    FC - 11111100 - forces the stack to its proper location
 *                    and locks down the firmware space.
 *
 * We also push "$F0" into FFEF, where the low nibble is a page number.
 * (The page number is usually 6 when loaded on a ///.  2000-9FFF is generally
 *  controlled by this flag, but we're not going to emulate that yet - which
 *  gives us a sort of stripped down, 64K Apple /// without the ROM.)
 *
 * FFD0 controls which 256-byte page of memory should be used for ZP (with
 * its alternate, i.e., EOR $01xx, as the stack).  We don't use this either,
 * although Star Emulator does write a $00 to it.  We also don't use FFDE or
 * FFEE, which Star Emulator zeroes out to shut SOS up.
 */

unsigned long colortab[16]={
 0x000000, 0xDD0033, 0x000099, 0xDD22DD, 
 0x007722, 0x555555, 0x2222FF, 0x66AAFF, 
 0x885500, 0xFF6600, 0xAAAAAA, 0xFF9988, 
 0x11DD00, 0xFFFF00, 0x44FF99, 0xFFFFFF
};

int DiskSlot;

enum DiskTypes
{
    UnknownType = 0,
    RawType     = 1,
    DOSType     = 2,
    ProDOSType  = 3,
    SimsysType  = 4,
    XgsType     = 5
};

struct DriveState
{
    /* variables for each disk */
    char DiskFN[ 1024 ]; /* MS-DOS filename of disk image */
    int DiskFH;        /* MS-DOS file handle of disk image */
    long DiskSize;     /* length of disk image file in bytes */
    enum DiskTypes DiskType; /* Type of disk image */
    int WritePro;      /* 0:write-enabled, !0:write-protected */
    int TrkBufChanged; /* Track buffer has changed */
    int TrkBufOld;     /* Data in track buffer needs updating before I/O */
    int ShouldRecal;
    /* variables used during emulation */
    int Track;    /* 0-70 */
    int Phase;    /* 0- 3 */
    int ReadWP;   /* 0/1  */
    int Active;   /* 0/1  */
    int Writing;  /* 0/1  */
} DrvSt[ 2 ];

int  CurDrv;
byte DataLatch;

int WriteAccess = 0; /* Flag indicates that the hardware I/O memory location
                        was written to. Use during writing data. */

unsigned char TrackBuffer[ 0x1a00 ];
unsigned int TrkBufIdx;

long tell (int handle) { return lseek (handle,0, SEEK_END);}
long FileSize( int filehandle )
{
    long filelen;
	filelen=lseek (filehandle,0L, SEEK_END);
	//printf("file, filelen:%d\n",filelen);
    	lseek( filehandle, 0L, SEEK_SET );
    	return filelen;
}

/* Make sure the track buffer index (given by idx) is within the correct
   range. This function should be called before every time you access the
   TrackBuffer. */
void RangeCheckTBI( unsigned int *idx )
{
    while ( *idx >= TrackBufLen )
    {
        *idx -= TrackBufLen;
    }
}

/* 6&2 data translation table
   from Peter Koch's emulator
   as in Andrew Gregory's emulator */
unsigned char translate[ 256 ] =
{
    0x96, 0x97, 0x9a, 0x9b, 0x9d, 0x9e, 0x9f, 0xa6,
    0xa7, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb2, 0xb3,
    0xb4, 0xb5, 0xb6, 0xb7, 0xb9, 0xba, 0xbb, 0xbc,
    0xbd, 0xbe, 0xbf, 0xcb, 0xcd, 0xce, 0xcf, 0xd3,
    0xd6, 0xd7, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde,
    0xdf, 0xe5, 0xe6, 0xe7, 0xe9, 0xea, 0xeb, 0xec,
    0xed, 0xee, 0xef, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6,
    0xf7, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00, 0x01,
    0x80, 0x80, 0x02, 0x03, 0x80, 0x04, 0x05, 0x06,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x07, 0x08,
    0x80, 0x80, 0x80, 0x09, 0x0a, 0x0b, 0x0c, 0x0d,
    0x80, 0x80, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
    0x80, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x1b, 0x80, 0x1c, 0x1d, 0x1e,
    0x80, 0x80, 0x80, 0x1f, 0x80, 0x80, 0x20, 0x21,
    0x80, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x29, 0x2a, 0x2b,
    0x80, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32,
    0x80, 0x80, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
    0x80, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f
};

void GotoHardSector( struct DriveState *ds, int sector )
{
    /* from DOS 3.3 Sector column in above table */
    int DOS33Skew[16]={0, 7, 14, 6, 13, 5, 12, 4, 11, 3, 10, 2, 9, 1, 8, 15};
    int ProDOSSkew[ 16 ] = { 0, 8, 1, 9, 2, 10, 3, 11, 4, 12, 5, 13, 6, 14, 7, 15 };

    lseek( ds->DiskFH, (long)( ds->Track >> 1 ) * 4096L, SEEK_SET );
    if ( ds->DiskType == DOSType )
    {
        lseek( ds->DiskFH, 256L * (long)DOS33Skew[ sector ], SEEK_CUR );
    }
    if ( ds->DiskType == ProDOSType )
    {
        lseek( ds->DiskFH, 256L * (long)ProDOSSkew[ sector ], SEEK_CUR );
    }
    if ( ds->DiskType == SimsysType )
    {
        lseek(ds->DiskFH, 256L*(long)ProDOSSkew[sector]+30, SEEK_CUR);
    }
    if ( ds->DiskType == XgsType ) /* currently only PO 2MG supported */
    {
        lseek(ds->DiskFH, 256L*(long)ProDOSSkew[sector]+64, SEEK_CUR);
    }
}

/* find and read a given hard sector given the disk format type */
/* only for .DO type files */
void ReadHardSector( struct DriveState *ds, int sector, unsigned char *buf )
{
    GotoHardSector( ds, sector );
    read( ds->DiskFH, buf, 256 );
}

/* find and write a given hard sector given the disk format type */
/* only for .DO type files */
void WriteHardSector( struct DriveState *ds, int sector, char *buf )
{
    GotoHardSector( ds, sector );
    write( ds->DiskFH, buf, 256 );
}

/* Sector=0..15 */
void NibbliseSector( unsigned char *data, unsigned char **trackbufptr,
                     int volume, int track, int sector )
{
    /* Define a template for all disk sectors */
    static unsigned char disktemplate[ 28 ] =
    {
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, /*  0- 5 = self-sync bytes */
        0xd5, 0xaa, 0x96,                   /*  6- 8 = address header */
        0, 0, 0, 0, 0, 0, 0, 0,             /*  9-16 = address */
        0xde, 0xaa, 0xeb,                   /* 17-19 = address trailer */
        0xff, 0xff, 0xff, 0xff, 0xff,       /* 20-24 = self-sync bytes */
        0xd5, 0xaa, 0xad                    /* 25-27 = data header */
    };

    int diskbyte, checksum, v;
    unsigned char *trackbuf;

    trackbuf = *trackbufptr;

    /* fill in address in template */
    checksum = volume ^ track ^ sector;
    disktemplate[  9 ] = ( volume   >> 1 ) | 0xaa;
    disktemplate[ 10 ] =   volume          | 0xaa;
    disktemplate[ 11 ] = ( track    >> 1 ) | 0xaa;
    disktemplate[ 12 ] =   track           | 0xaa;
    disktemplate[ 13 ] = ( sector   >> 1 ) | 0xaa;
    disktemplate[ 14 ] =   sector          | 0xaa;
    disktemplate[ 15 ] = ( checksum >> 1 ) | 0xaa;
    disktemplate[ 16 ] =   checksum        | 0xaa;

    /* template */
    for ( diskbyte = 0; diskbyte < 28; diskbyte++ )
    {
        *trackbuf++ = disktemplate[ diskbyte ];
    }
    /* data */
    checksum = 0;
    for ( diskbyte = 0; diskbyte < 0x156; diskbyte++ )
    {
        v = ( diskbyte >= 0x56 ) ?
            /* get 6-bit byte */
            data[ diskbyte - 0x56 ] >> 2 :
            /* build 6-bit byte from 3 x 2 bits */
            ( ( data[ diskbyte        ] & 0x02 ) >> 1 ) |
            ( ( data[ diskbyte        ] & 0x01 ) << 1 ) |
            ( ( data[ diskbyte + 0x56 ] & 0x02 ) << 1 ) |
            ( ( data[ diskbyte + 0x56 ] & 0x01 ) << 3 ) |
            ( ( data[ diskbyte + 0xac ] & 0x02 ) << 3 ) |
            ( ( data[ diskbyte + 0xac ] & 0x01 ) << 5 );
        *trackbuf++ = translate[ ( checksum ^ v ) & 0x3f ];
        checksum = v;
    }

    /* data checksum */
    *trackbuf++ = translate[ checksum & 0x3f ];
    /* data trailer */
    *trackbuf++ = 0xde;
    *trackbuf++ = 0xaa;
    *trackbuf++ = 0xeb;

    *trackbufptr = trackbuf;
}

/* Nibblise Track: Convert from DOS 3.3 sector-level data into a raw Disk ][
   track-level byte stream */
void NibbliseTrack( struct DriveState *ds )
{
    unsigned char SectorBuffer[ 258 ];
    unsigned char *TrackBufPtr = TrackBuffer;
    int idx;

    /* zero-out unused buffer space */
    SectorBuffer[ 256 ] = 0;
    SectorBuffer[ 257 ] = 0;
    /* encode each hard sector */
    for ( idx = 0; idx < 16; idx++ )
    {
        ReadHardSector( ds, idx, SectorBuffer );
        NibbliseSector( SectorBuffer, &TrackBufPtr, 254, ds->Track >> 1, idx );
    }
}

void ReadTrack( struct DriveState *ds )
{
    int idx;

    /* Make sure that any unused part of the buffer has 0xff's in it */
    for ( idx = 0; idx < 0x1a00; idx++ )
    {
        TrackBuffer[ idx ] = 0xff;
    }
    if ( ds->DiskType == RawType )
    {
        /* Disk ][ track/byte format (0x1a00 bytes/track) - just read the bytes */
        lseek( ds->DiskFH, (long)( ds->Track >> 1 ) * 0x1a00L, SEEK_SET );
        read( ds->DiskFH, TrackBuffer, 0x1a00 );
    }
    if ( ds->DiskType == DOSType || ds->DiskType == ProDOSType
      || ds->DiskType == SimsysType || ds->DiskType == XgsType )
    {
        /* Track/sector format (4096 bytes/track) - translate to Disk ][ format */
        NibbliseTrack( ds );
    }
    /* new track - so clear changed flag */
    ds->TrkBufChanged = 0;
    ds->TrkBufOld = 0;
    ds->ShouldRecal = 0;
}

/* return character offset bytes from idx. Wrap around the end of the buffer
   if required */
unsigned char GetByte( unsigned int idx, unsigned int offset )
{
    unsigned int i = idx + offset;
    RangeCheckTBI( &i );
    return TrackBuffer[ i ];
}

void DeNibbliseData( int idx, char *SectorBuffer )
{
    unsigned char data, v;
    int diskbyte;

    /* idx is the offset of the start of the disk data */
    v = 0;
    for ( diskbyte = 0; diskbyte < 0x156; diskbyte++ )
    {
        data = translate[ GetByte( idx, diskbyte ) ];
        if ( diskbyte < 0x56 )
        {
            /* turn 6 bits into 3 lots of 2 bits */
            data ^= v;
            SectorBuffer[ diskbyte        ]  = ( data & 0x01 ) << 1;
            SectorBuffer[ diskbyte        ] |= ( data & 0x02 ) >> 1;
            SectorBuffer[ diskbyte + 0x56 ]  = ( data & 0x04 ) >> 1;
            SectorBuffer[ diskbyte + 0x56 ] |= ( data & 0x08 ) >> 3;
            SectorBuffer[ diskbyte + 0xac ]  = ( data & 0x10 ) >> 3;
            SectorBuffer[ diskbyte + 0xac ] |= ( data & 0x20 ) >> 5;
            v = data;
        }
        else {
            /* get 6 more bits */
            data ^= v;
            SectorBuffer[ diskbyte - 0x56 ] |= ( data << 2 ) & 0xfc;
            v = data;
        }
    }
}

/* De-Nibblise Track: Convert from raw Disk ][ disk byte stream into DOS 3.3
   compatible track/sector-level data */
#pragma argsused
void DeNibbliseTrack( struct DriveState *ds )
{
    unsigned int idx;       /* index into track buffer */
    unsigned int start;     /* index to start of track data */
    unsigned int rotations; /* # times the disk has 'rotated' */
    int volume, track, sector, chksum, Done;
    char SectorBuffer[ 258 ];

    start = 0xffff;
    idx = 0;
    /* keep searching until idx == start */
    Done = 0;
    rotations = 0;
    while ( !Done ) {
        if ( idx >= TrackBufLen )
        {
            rotations++;
        }
        if ( GetByte( idx, 0 ) == 0xd5 &&
             GetByte( idx, 1 ) == 0xaa &&
             GetByte( idx, 2 ) == 0x96 )
        {
            /* Found address header */
            volume = ( ( ( GetByte( idx, 3 ) << 1 ) & 0xff ) | 0x55 ) & GetByte( idx,  4 );
            track  = ( ( ( GetByte( idx, 5 ) << 1 ) & 0xff ) | 0x55 ) & GetByte( idx,  6 );
            sector = ( ( ( GetByte( idx, 7 ) << 1 ) & 0xff ) | 0x55 ) & GetByte( idx,  8 );
            chksum = ( ( ( GetByte( idx, 9 ) << 1 ) & 0xff ) | 0x55 ) & GetByte( idx, 10 );
            if ( chksum == ( volume ^ track ^ sector ) )
            {
                /* checksum is valid */
                /* NOTE: DOS 3.3 and ProDOS chop off the last byte (0xeb)
                   before it is written. A proper trailer should be DE AA EB
                   instead only DE AA is written. */
                if ( GetByte( idx, 11 ) == 0xde &&
                     GetByte( idx, 12 ) == 0xaa )
                {
                    /* valid address trailer */
                    /* skip over self-sync bytes */
                    idx += 14;
                    if ( idx >= TrackBufLen )
                    {
                        rotations++;
                    }
                    RangeCheckTBI( &idx );
                    while ( GetByte( idx, 0) != 0xd5 && idx < TrackBufLen )
                    {
                        idx++;
                    }
                    /* check for data header */
                    if ( GetByte( idx, 0 ) == 0xd5 &&
                         GetByte( idx, 1 ) == 0xaa &&
                         GetByte( idx, 2 ) == 0xad )
                    {
                        /* valid data header */
                        idx += 3;
                        if ( idx >= TrackBufLen )
                        {
                            rotations++;
                        }
                        RangeCheckTBI( &idx );
                        if ( idx == start )
                        {
                            Done = 1;
                            continue;
                        }
                        if ( start == 0xffff )
                        {
                            start = idx;
                        }
                        /* get, decode and write data */
                        DeNibbliseData( idx, SectorBuffer );
                        WriteHardSector( ds, sector, SectorBuffer );
                    }
                }
            }
        }
        idx++;
        /* Check timeout */
        if ( rotations > 32 )
        {
            Done = 1;
        }
    }
}

void WriteTrack( struct DriveState *ds )
{
   int idx;

   /* fill any unused space in buffer with 0xff's */
   if ( TrackBufLen < 0x1a00 )
   {
       for ( idx = TrackBufLen; idx < 0x1a00; idx++ )
       {
           TrackBuffer[ idx ] = 0xff;
       }
   }

   if ( ds->DiskType == RawType )
   {
       /* Disk ][ track/byte format (0x1a00 bytes/track) - just write the bytes */
       lseek( ds->DiskFH, (long)( ds->Track >> 1 ) * 0x1a00L, SEEK_SET );
       write( ds->DiskFH, TrackBuffer, 0x1a00 );
   }
   if ( ds->DiskType == DOSType || ds->DiskType == ProDOSType
     || ds->DiskType == SimsysType || ds->DiskType == XgsType )
   {
       /* Track/sector format (4096 bytes/track) - translate from Disk ][ format */
       DeNibbliseTrack( ds );
   }
   ds->DiskSize = FileSize( ds->DiskFH );
   if (ds->DiskSize<143360) ds->DiskSize=143360; // uso. 2002.1109
   /* track has been saved - so clear changed flag */
   ds->TrkBufChanged = 0;
}

byte ReadDiskIO( word Address )
{
    struct DriveState *ds;
    int newtrack;
    int writezero;
    char t[ 20 ];

    ds = &DrvSt[ CurDrv ];
    newtrack = ds->Track;

    /* Update track buffer */
    if ( /*ds->Active &&*/ ds->Track % 2 == 0 && !FastMode )
    {
        /* Only update when the disk drive is active AND the track is valid
           AND we're emulating correct disk drive speed */
        Diff = ClockTick - LastIO;
        if ( Diff == 31L )
        {
            Diff = 32L; /* handle faster 65C02 execution */
        }
        LeftOverCycles = Diff;
        writezero = 0;
        while ( LeftOverCycles >= 32L )
        {
            if ( ds->Writing && writezero )
            {
                RangeCheckTBI( &TrkBufIdx );
                TrackBuffer[ TrkBufIdx ] = 0;
            }
            TrkBufIdx++;
            LeftOverCycles -= 32L;
            writezero = 1; /* delays >32us cause 0's to be written */
        }
    }

    /* Handle I/O access */
    switch (Address&0x0f) {
    case 0x00: /* Q0 - Phase 0 off */
    case 0x02: /* Q1 - Phase 1 off */
    case 0x04: /* Q2 - Phase 2 off */
    case 0x06: /* Q3 - Phase 3 off */
        break;
    case 0x01:
        /* Q0 - Phase 0 on */
        if ( ds->Active )
        {
            if ( ds->Phase == 1 )
            {
                newtrack--; /* move head out */
            }
            if ( ds->Phase == 3 )
            {
                newtrack++; /* move head in  */
            }
            ds->Phase = 0;
        }
        break;
    case 0x03:
        /* Q1 - Phase 1 on */
        if ( ds->Active )
        {
            if ( ds->Phase == 2 )
            {
                newtrack--; /* move head out */
            }
            if ( ds->Phase == 0 )
            {
                newtrack++; /* move head in  */
            }
            ds->Phase = 1;
        }
        break;
    case 0x05:
        /* Q2 - Phase 2 on */
        if ( ds->Active )
        {
            if ( ds->Phase == 3 )
            {
                newtrack--; /* move head out */
            }
            if ( ds->Phase == 1 )
            {
                newtrack++; /* move head in  */
            }
            ds->Phase = 2;
        }
        break;
    case 0x07:
        /* Q3 - Phase 3 on */
        if ( ds->Active )
        {
            if ( ds->Phase == 0 )
            {
                newtrack--; /* move head out */
            }
            if ( ds->Phase == 2 )
            {
                newtrack++; /* move head in  */
            }
            ds->Phase = 3;
        }
        break;
    case 0x08:
        /* Q4 - Drive off */
        ds->Active = 0;
        break;
    case 0x09:
        /* Q4 - Drive on */
        ds->Active = 1;
        break;
    case 0x0a:
        /* Q5 - Drive 1 select */
        if ( CurDrv != 0 )
        {
            if ( ds->TrkBufChanged && ds->Track % 2 == 0 )
            {
                WriteTrack( ds );
            }
            CurDrv = 0;
            ds = &DrvSt[ CurDrv ];
            ReadTrack( ds );
            TrkBufIdx = 0;
            newtrack = ds->Track;
            if ( DrvSt[ 1 - CurDrv ].Active )
            {
                ds->Active = 1;
            }
            DrvSt[ 1 - CurDrv ].Active = 0; /* force other disk drive to be inactive */
        }
        break;
    case 0x0b:
        /* Q5- Drive 2 select */
        if ( CurDrv != 1 )
        {
            if ( ds->TrkBufChanged && ds->Track % 2 == 0 )
            {
                WriteTrack( ds );
            }
            CurDrv = 1;
            ds = &DrvSt[ CurDrv ];
            ReadTrack( ds );
            TrkBufIdx = 0;
            newtrack = ds->Track;
            if ( DrvSt[ 1 - CurDrv ].Active )
            {
                ds->Active = 1;
            }
            DrvSt[ 1 - CurDrv ].Active = 0; /* force other disk drive to be inactive */
        }
        break;
    case 0x0c:
    case 0x0d:
    case 0x0e:
    case 0x0f:
        if ( ds->TrkBufOld && ds->Track % 2 == 0 )
        {
            ReadTrack( ds );
            TrkBufIdx = 0;
        }
        /* handle switch changes first */
        switch ( Address & 0x0f )
        {
        case 0x0c:
            /* Q6 off */
            ds->ReadWP = 0;
            break;
        case 0x0d:
            /* Q6 on */
            ds->ReadWP = 1;
            break;
        case 0x0e:
            /* Q7 off */
            if ( ds->Writing )
            {
                /* If <32us have passed since writing a byte, the byte just
                   written will not be complete written. Place an 0xff there
                   and move on. */
                if ( Diff < 32L )
                {
                    RangeCheckTBI( &TrkBufIdx );
                    TrackBuffer[ TrkBufIdx ] = 0xff;
                    ds->TrkBufChanged = 1;
                    Diff = 32L;
                    LeftOverCycles = 0L;
                }
            }
            ds->Writing = 0;
            break;
        case 0x0f:
            /* Q7 on */
            if ( !ds->Writing )
            {
                /* If <32us have passed since reading a byte, the byte just
                   read will be (partially) overwritten. Overwrite the whole
                   lot. */
                LeftOverCycles = 0L;
            }
            ds->Writing = 1;
            break;
        }
        /* Make sure the disk drive is active (motor on) before doing anything */
        if ( !ds->Active )
        {
            /* ProDOS appears to always want some changing data appear in the
               data latch, so generate some garbage to feed it! */
            if ( !WriteAccess )
            {
               DataLatch = ClockTick & 0xff;
            }
            break;
        }
        /* then handle the mode */
        if ( ds->Writing )
        {
            if ( !ds->WritePro && WriteAccess && ds->Track % 2 == 0 )
            {
                /* write disk byte */
                if (FastMode) {
                    RangeCheckTBI( &TrkBufIdx );
                    TrackBuffer[ TrkBufIdx++ ] = DataLatch;
                    ds->TrkBufChanged = 1;
                } else {
                    if ( Diff >= 32L )
                    {
                        if ( Diff <= 40L )
                        {
                            LeftOverCycles = 0L;
                        }
                        LastIO = ClockTick - LeftOverCycles;
                    }
                    RangeCheckTBI( &TrkBufIdx );
                    TrackBuffer[ TrkBufIdx ] = DataLatch;
                    ds->TrkBufChanged = 1;
                }
            }
        }
        else {
            /* read */
            if ( ds->ReadWP )
            {
                /* get write-protect status: bit 7 set=write-protected */
                DataLatch = ds->WritePro ? 0xff : 0x00;
            }
            else
            {
                /* read disk byte */
                if (FastMode) {
                    RangeCheckTBI( &TrkBufIdx );
                    DataLatch = TrackBuffer[ TrkBufIdx++ ];
                } else {
                    if ( Diff >= 32L )
                    {
                        RangeCheckTBI( &TrkBufIdx );
                        DataLatch = TrackBuffer[ TrkBufIdx ];
                        LastIO = ClockTick - LeftOverCycles;
                    }
                    else
                    {
                        DataLatch = 0;
                    }
                }
            }
        }
        break;
    }

    if (ds->Track != newtrack) {
        /* Disk tracks have changed */
        /* Make sure current track is in range */
        if ( newtrack < 0 )
        {
            newtrack = 0;
            ds->ShouldRecal = 1;
        }
        if ( newtrack > 70 )
        {
            newtrack = 70;
            ds->ShouldRecal = 1;
        }
        /* NOTE: only even Disk ][ tracks are valid */
        if ( ds->TrkBufChanged && ds->Track % 2 == 0 )
        {
            WriteTrack( ds );
        }
        ds->Track = newtrack;
        ds->TrkBufOld = 1;
    }

    return DataLatch;
}

void WriteDiskIO( word Address, byte Data )
{
    if (cpuspeed==2)
    {
     printf ("WARNING: Trying to address the drive with CPU clock-doubled\n");
    }

    /* Address = 0xe0-0xef */
    Address &= 0x0f; /* just keep offset */

    WriteAccess = 1;
    if ( ( Address & 0x0c ) == 0x0c )
    {
        DataLatch = Data;
    }
    ReadDiskIO( Address );
    WriteAccess = 0;
}

void DiskReset( void )
{
    DrvSt[ 0 ].Active = 0;
    DrvSt[ 0 ].ReadWP = 0;
    DrvSt[ 0 ].Writing  =0;
    DrvSt[ 1 ].Active = 0;
    DrvSt[ 1 ].ReadWP = 0;
    DrvSt[ 1 ].Writing = 0;
}

void UnmountDisk( int disk )
{
    struct DriveState *ds;
    ds = &DrvSt[ disk ];
    if ( ds->TrkBufChanged )
    {
        WriteTrack( ds );
    }
    close( ds->DiskFH );
    if ( ds->DiskSize == 0L )
    {
        /* nothing in the file - might as well delete it */
        //remove ( ds->DiskFN );
    }
    ds->DiskFH = 0;
}

void DiskAutoID( struct DriveState *ds )
{
    unsigned char t1, t2; /* temporary data during identification */
    int idcount; /* count of ProDOS identifying characteristics */

    /* if the type is unknown, try to auto-identify */
    if ( ds->DiskType == UnknownType )
    {
//      if ( ds->DiskSize % 0x1a00L == 0L )   /* Raw 0x1a00 bytes/track */
        if ( ds->DiskSize == 143390 )
            ds->DiskType = SimsysType;
        else if ( ds->DiskSize == 143424 )
            ds->DiskType = XgsType;
        else if ( ds->DiskSize >= 200000 )
            ds->DiskType = RawType;
        else
            ds->DiskType = DOSType;
    }
}

void MountDisk( int disk )
{
	//printf("PRING GOD DAMN YOU !\n");
    struct DriveState *ds;
//    unsigned int attr; /* MS-DOS file attributes */

    ds = &DrvSt[ disk ];

    /* open disk file and set disk parameters */
//    attr = GetAttrib( ds->DiskFN );
//    if ( attr == 0xffff )
//    {
//        attr = 0;
//    }
//    ds->WritePro = ( attr & FA_RDONLY ) ? 1 : 0;
    ds->WritePro=0;
    ds->DiskFH = open( ds->DiskFN,
#ifdef __MSDOS__
                                  O_BINARY|
#endif
                                   (ds->WritePro ? O_RDONLY : O_RDWR) );
    ds->DiskSize = 0L;
    if ( ds->DiskFH >= 0 )
    {
        ds->DiskSize = FileSize( ds->DiskFH );
        if (ds->DiskSize<143360) ds->DiskSize=143360; // uso. 2002.1109
    }
    DiskAutoID( ds );
    ds->TrkBufChanged = 0;
    ds->TrkBufOld = 1;
    ds->ShouldRecal = 1;
    /* set disk emulation parameters */
    ds->Track = 0;
    ds->Phase = 0;
    ds->ReadWP = 0;
    ds->Active = 0;
    ds->Writing = 0;
}

void InitDisk( int slot )
{
    DiskSlot = slot;

    strcpy( DrvSt[ 0 ].DiskFN, "disk6a.dsk" );
    DrvSt[ 0 ].DiskType = UnknownType;
    MountDisk( 0 );
    strcpy( DrvSt[ 1 ].DiskFN, "disk6b.dsk" );
    DrvSt[ 1 ].DiskType = UnknownType;
    MountDisk( 1 );

    CurDrv = 0;
    DataLatch = 0;
    TrkBufIdx = 0;
}

void ShutdownDisk( void )
{
    UnmountDisk( 0 );
    UnmountDisk( 1 );
}

void safepset (word x, word y, unsigned long c)
{
 if (curbuf[x][y]==c) return;
 curbuf[x][y]=c;
 putpixel(screen, x, y, c);
}

void flushscreen (void)
{
 word x,y;

 for (y=0; y<480; y++)
  for (x=0; x<640; x++)
  {
   curbuf[x][y]=0;
   putpixel(screen,x,y,0);
  }
}

void gputch40c(byte x, byte y, byte c, byte a)
{
 word xx, yy, cc, i;
 byte f, b;

 if (real3&&hassysfont)
  myfont=sysfont;
 else if (!hasstarfont)
  myfont=sysfont;
 else
  myfont=starfont;

 f=(a>>4)&0x0F;
 b=a&0x0F;

 i=(c<128)?0xFF:0x00;
 cc=c&127;
 cc<<=3;
 for (yy=0; yy<8; yy++)
 {
  byte offset;

  for (offset=48; offset<50; offset++)
  {
   byte ccc;

   ccc=myfont[cc+yy];
   if (i) ccc=255-ccc;

   safepset(2*(x*7)+40, ((yy+(y*8))*2)+offset, ccc& 1?colortab[f]:colortab[b]);
   safepset(2*(x*7)+42, ((yy+(y*8))*2)+offset, ccc& 2?colortab[f]:colortab[b]);
   safepset(2*(x*7)+44, ((yy+(y*8))*2)+offset, ccc& 4?colortab[f]:colortab[b]);
   safepset(2*(x*7)+46, ((yy+(y*8))*2)+offset, ccc& 8?colortab[f]:colortab[b]);
   safepset(2*(x*7)+48, ((yy+(y*8))*2)+offset, ccc&16?colortab[f]:colortab[b]);
   safepset(2*(x*7)+50, ((yy+(y*8))*2)+offset, ccc&32?colortab[f]:colortab[b]);
   safepset(2*(x*7)+52, ((yy+(y*8))*2)+offset, ccc&64?colortab[f]:colortab[b]);

   safepset(2*(x*7)+41, ((yy+(y*8))*2)+offset, ccc& 1?colortab[f]:colortab[b]);
   safepset(2*(x*7)+43, ((yy+(y*8))*2)+offset, ccc& 2?colortab[f]:colortab[b]);
   safepset(2*(x*7)+45, ((yy+(y*8))*2)+offset, ccc& 4?colortab[f]:colortab[b]);
   safepset(2*(x*7)+47, ((yy+(y*8))*2)+offset, ccc& 8?colortab[f]:colortab[b]);
   safepset(2*(x*7)+49, ((yy+(y*8))*2)+offset, ccc&16?colortab[f]:colortab[b]);
   safepset(2*(x*7)+51, ((yy+(y*8))*2)+offset, ccc&32?colortab[f]:colortab[b]);
   safepset(2*(x*7)+53, ((yy+(y*8))*2)+offset, ccc&64?colortab[f]:colortab[b]);
  }
 }
}

void gputch40bw(byte x, byte y, byte c)
{
 gputch40c(x,y,c,0xF0);
}

void gputch80(byte x, byte y, byte c)
{
 word xx, yy, cc, i;

 if (real3&&hassysfont)
  myfont=sysfont;
 else if (!hasstarfont)
  myfont=sysfont;
 else
  myfont=starfont;

 i=(c<128);
 cc=c&127;
 cc<<=3;
 for (yy=0; yy<8; yy++)
 {
  byte offset;

  for (offset=48; offset<50; offset++)
  {
   byte ccc;

   ccc=myfont[cc+yy];
   if (i) ccc=255-ccc;

   safepset((x*7)+40, ((yy+(y*8))*2)+offset, ccc& 1?0xFFFFFF:0x000000);
   safepset((x*7)+41, ((yy+(y*8))*2)+offset, ccc& 2?0xFFFFFF:0x000000);
   safepset((x*7)+42, ((yy+(y*8))*2)+offset, ccc& 4?0xFFFFFF:0x000000);
   safepset((x*7)+43, ((yy+(y*8))*2)+offset, ccc& 8?0xFFFFFF:0x000000);
   safepset((x*7)+44, ((yy+(y*8))*2)+offset, ccc&16?0xFFFFFF:0x000000);
   safepset((x*7)+45, ((yy+(y*8))*2)+offset, ccc&32?0xFFFFFF:0x000000);
   safepset((x*7)+46, ((yy+(y*8))*2)+offset, ccc&64?0xFFFFFF:0x000000);
  }
 }
}

void gputb40c(byte x, byte y, byte d, byte a)
{
 /*
  * (x*7)+0 (x*7)+1 (x*7)+2 (x*7)+3 (x*7)+4 (x*7)+5 (x*7)+6
  *  d&0x01  d&0x02  d&0x04  d&0x08  d&0x10  d&0x20  d&0x40
  */

 byte f, b;
 word xx;
 int offset;

 f=(a>>4)&0x0F;
 b=a&0x0F;

 for (offset=48; offset<50; offset++)
 {
  safepset((x*14)+40, (y*2)+offset, (d&0x01)?colortab[f]:colortab[b]);
  safepset((x*14)+42, (y*2)+offset, (d&0x02)?colortab[f]:colortab[b]);
  safepset((x*14)+44, (y*2)+offset, (d&0x04)?colortab[f]:colortab[b]);
  safepset((x*14)+46, (y*2)+offset, (d&0x08)?colortab[f]:colortab[b]);
  safepset((x*14)+48, (y*2)+offset, (d&0x10)?colortab[f]:colortab[b]);
  safepset((x*14)+50, (y*2)+offset, (d&0x20)?colortab[f]:colortab[b]);
  safepset((x*14)+52, (y*2)+offset, (d&0x40)?colortab[f]:colortab[b]);

  safepset((x*14)+41, (y*2)+offset, (d&0x01)?colortab[f]:colortab[b]);
  safepset((x*14)+43, (y*2)+offset, (d&0x02)?colortab[f]:colortab[b]);
  safepset((x*14)+45, (y*2)+offset, (d&0x04)?colortab[f]:colortab[b]);
  safepset((x*14)+47, (y*2)+offset, (d&0x08)?colortab[f]:colortab[b]);
  safepset((x*14)+49, (y*2)+offset, (d&0x10)?colortab[f]:colortab[b]);
  safepset((x*14)+51, (y*2)+offset, (d&0x20)?colortab[f]:colortab[b]);
  safepset((x*14)+53, (y*2)+offset, (d&0x40)?colortab[f]:colortab[b]);
 }
}

void gputb40bw(byte x, byte y, byte d)
{
 gputb40c(x,y,d,0xF0);
}

void gputb80(byte x, byte y, byte d, byte bank)
{
 safepset((x*14)+40+bank, (y*2)+48, (d&0x01)?0xFFFFFF:0x000000);
 safepset((x*14)+42+bank, (y*2)+48, (d&0x02)?0xFFFFFF:0x000000);
 safepset((x*14)+44+bank, (y*2)+48, (d&0x04)?0xFFFFFF:0x000000);
 safepset((x*14)+46+bank, (y*2)+48, (d&0x08)?0xFFFFFF:0x000000);
 safepset((x*14)+48+bank, (y*2)+48, (d&0x10)?0xFFFFFF:0x000000);
 safepset((x*14)+50+bank, (y*2)+48, (d&0x20)?0xFFFFFF:0x000000);
 safepset((x*14)+52+bank, (y*2)+48, (d&0x40)?0xFFFFFF:0x000000);
}

void wrtty (word Addr, byte Value)
{
 if ((Addr>=0x0400)&&(Addr<0x0800))
 {
  byte x, y, block1, block2;
  
  /* XXX: doesn't deal with the softswitches properly */
  block1=(Addr&0x007F);
  block2=(Addr&0x0380)>>7; // # of 128s
  y=((block1/40)<<3)+block2;
  if (block1<120)
  {
   x=block1%40;

   if (tty_dblres)
   {
    gputch80(x*2,y,Value);
   }
   else if (tty_color && (!tty_page2))
   {
    gputch40c(x,y,Value,mem[Addr+0x0400]);
   }
   else if (tty_color && tty_page2)
   {
    gputch40c(x,y,mem[Addr+0x0400],Value);
   }
   else if ((!tty_page2)&&(!tty_color))
   {
    gputch40bw(x,y,Value);
   }
  }
 }
 else if ((Addr>=0x0800)&&(Addr<0x0C00))
 {
  byte x, y, block1, block2;
  
  /* XXX: doesn't deal with the softswitches properly */
  block1=(Addr&0x007F);
  block2=(Addr&0x0380)>>7; // # of 128s
  y=((block1/40)<<3)+block2;
  if (block1<120)
  {
   x=block1%40;

   if (tty_dblres)
   {
    gputch80((x*2)+1,y,Value);
   }
   else if (tty_color && tty_page2)
   {
    gputch40c(x,y,Value,mem[Addr-0x0400]);
   }
   else if (tty_color && (!tty_page2))
   {
    gputch40c(x,y,mem[Addr-0x0400],Value);
   }
   else if (tty_page2 && (!tty_color))
   {
    gputch40bw(x,y,Value);
   }
  }
 }
}

void wrttyg(word Addr, byte Value)
{
 word eights, threes, forties, ones;

 ones=((Addr&0x1FFF)/1024);
 threes=(Addr%1024);
 eights=(threes/128);
 forties=(threes%128);
 if (forties<120)
 {
  byte x;
  word y;

  x=forties%40;
  y=((forties/40)*64)+(eights*8)+ones;

  if ((Addr>=0x2000)&&(Addr<0x4000))
  {
   if (tty_dblres)
    gputb80(x, y, Value, 0);
   else if ((!tty_color)&&(!tty_page2))
    gputb40bw(x, y, Value);
   else if (tty_page2&&tty_color)
    gputb40c(x, y, mem[Addr+0x2000], Value);
   else if (tty_color&&(!tty_page2))
    gputb40c(x, y, Value, mem[Addr+0x2000]);
  }
  else if ((Addr>=0x4000)&&(Addr<0x6000))
  {
   if (tty_dblres)
    gputb80(x, y, Value, 1);
   else if ((!tty_color)&&tty_page2)
    gputb40bw(x, y, Value);
   else if (tty_page2&&tty_color)
    gputb40c(x, y, Value, mem[Addr-0x2000]);
   else if (tty_color&&(!tty_page2))
    gputb40c(x, y, mem[Addr-0x2000], Value);
  }
 }
}

void tty_do (byte a)
{
 word t;
 byte oldset, newset;

 oldset=(tty_color?0x1:0x0)|(tty_dblres?0x2:0x0)|(tty_page2?0x4:0x0)|(tty_grx?0x8:0x0);
 
 switch (a)
 {
  case 0:
   tty_color=0;
   break;
  case 1:
   tty_color=1;
   break;
  case 2:
   tty_dblres=0;
   break;
  case 3:
   tty_dblres=1;
   break;
  case 4:
   tty_page2=0;
   break;
  case 5:
   tty_page2=1;
   break;
  case 6:
   tty_grx=0;
   break;
  case 7:
   tty_grx=1;
   break;
 }

 if (tty_grx)
  for (t=0x2000; t<0x6000; t++) wrttyg(t, mem[t]);
 else
  for (t=0x0400; t<0x0C00; t++) wrtty(t, mem[t]);
 
}

byte Rd6502(register word Addr)
{
 /* XXX: hardware accesses in C0xx */

 /* Implement Floating ZP & Stack */
 if (Addr<0x0200)
 {
  if (real3)
  {
   if ((Addr&0xFF00)==0) return mem[((mem[0xFFD0])<<8)+Addr];
   if (fstack)
   {
    return mem[(((mem[0xFFD0])^1)<<8)+(Addr&0xFF)];
   }
  }
 }

 if ((Addr&0xF000)==0xC000)
 {
  if (hideio) return mem[Addr];
 }
 
 if ((Addr&0xFFF0)==0xC000)
 {
  if ((Addr&0xF)<8) return lastkey;

  /*
   * According to Apple:
   *  128 "128" bit of the key presset.  i.e., always off.
   *   64 "Doesn't know if it's in /// or ][ mode yet." i.e., always off.
   *   32 Right Alt is down
   *   16 Left Alt is down
   *    8 CAPS LOCK is down
   *    4 Ctrl is down
   *    2 Shift is down
   *    1 A key is pressed
   *
   * Also: Ctrl-Reset does an "NMI" interrupt, not a "RESET" interrupt.
   */

  return (key_shifts&KB_MENU_FLAG?0:32)|
         (key_shifts&KB_ALT_FLAG?0:16)|
         (key_shifts&KB_CAPSLOCK_FLAG?0:8)|
         (key_shifts&KB_CTRL_FLAG?0:4)|
         (key_shifts&KB_SHIFT_FLAG?0:2)|
         (keypressed()?1:0);
  
  return 0x7D; /* XXX: key modifiers? - this is what MESS reports. */
 }
 if ((Addr&0xFFF0)==0xC010)
 {
  lastkey&=0x7F;
  return lastkey;
 }
 if ((Addr&0xFFF0)==0xC050)
 {
  tty_do(Addr&0x0F);
  return 0x00; // xxx
 }
 if ((Addr&0xFFF0)==0xC0E0) return ReadDiskIO(Addr);

 if ((Addr&0xF000)==0xF000)
 {
  if ((Addr<0xFFD0)||(Addr>0xFFEF))
   if (isrom&&real3) return rom[Addr&0x0FFF];
 }
 
 return mem[Addr];
}

void Wr6502(register word Addr, register byte Value)
{
 if (Addr==0xFFD0)
 {
  mem[0xFFD0]=Value;
  printf ("Zero Page bank set to %02X.  Floating Stack bank set to %02X.\n", Value, Value^1);
  return;
 }

 /* Implement Floating ZP & Stack */
 if (Addr<0x0200)
 {
  if (real3)
  {
   if ((Addr&0xFF00)==0)
   {
    mem[((mem[0xFFD0])<<8)+Addr]=Value;
    return;
   }
   if (fstack)
   {
    mem[(((mem[0xFFD0])^1)<<8)+(Addr&0xFF)]=Value;
    return;
   }
  }
 }


 if ((Addr&0xF000)==0xC000)
 {
  if (hideio)
  {
   mem[Addr]=Value;
  }
 }

 if (tty_grx)
 {
  if ((Addr>=0x2000)&&(Addr<0x6000)) wrttyg(Addr, Value);
 }
 else
 {
  if ((Addr>=0x0400)&&(Addr<0x0C00)) wrtty(Addr, Value);
 }

 if ((Addr&0xFFF0)==0xC050)
 {
  tty_do(Addr&0x0F);
  return;
 }
 if ((Addr&0xFFF0)==0xC0E0)
 {
  WriteDiskIO(Addr,Value);
  return;
 }

 if ((Addr<0xC000)||((Addr&0xFFF0)==0xFFD0))
 {
  if (Addr==0xFFDF)
  {
   int old;
   old=mem[0xFFDF];
   if ((old&0x80)!=(Value&0x80))
   {
    cpuspeed=(Value&0x80)?1:2;
    printf ("6502 speed is now %d MHz.\n", cpuspeed); 
    CPU.IPeriod=65*cpuspeed;
   }
// if ((old&0x40)!=(Value&0x40)) { printf ("Unemulated change of mapped i/o to %s.\n", (Value&0x40)?"Enabled":"Disabled"); }
   if ((old&0x40)!=(Value&0x40))
   {
    if (Value&0x40)
    {
     hideio=0;

     printf ("Mapped I/O is now enabled.\n");
    }
    else
    {
     hideio=1;

     printf ("Mapped I/O is now hidden.\n");
    }
   }
   if ((old&0x20)!=(Value&0x20)) { printf ("Unemulated change of the screen to %s.\n", (Value&0x20)?"Enabled":"Disabled"); }
   if ((old&0x10)!=(Value&0x10)) { printf ("Unemulated change of the Reset button to %s.\n", (Value&0x10)?"Enabled":"Disabled"); }
   if ((old&0x08)!=(Value&0x08)) { printf ("High memory write is now %s.\n", (Value&0x08)?"Disabled":"Enabled"); }
// if ((old&0x04)!=(Value&0x04)) { printf ("Unemulated change of floating stack to %s.\n", (Value&0x04)?"Disabled":"Enabled"); }
   if ((old&0x04)!=(Value&0x04))
   {
    fstack=(Value&0x04)?0:1;
    printf ("The floating stack is now %s.\n", fstack?"enabled":"disabled");
   }
// if ((old&0x02)!=(Value&0x02)) { printf ("Unemulated change of ROM presence to %s.\n", (Value&0x02)?"Enabled":"Disabled"); }
   if ((old&0x02)!=(Value&0x02))
   {
    if (real3)
    {
     isrom=(Value&0x02);
     printf ("The ROM is now %svisible.\n", (Value&0x02)?"":"in");
    }
    else
    {
     printf ("Cannot %s the ROM because it is not loaded.\n", (Value&0x02)?"enable":"disable");
    }
   }

   if ((old&0x01)!=(Value&0x01)) { printf ("Unemulated change of ROM bank to %d.\n", (Value&0x80)?2:1); }
  }

  mem[Addr]=Value;
  return;
 }

 if (!(mem[0xFFDF]&0x08))
 {
  mem[Addr]=Value;
  return;
 }
}

void die (void)
{
 corpse++;
}
END_OF_FUNCTION(die)

byte Loop6502(register M6502 *R)
{
 if (corpse) return INT_QUIT;

 if (keypressed())
 {
  int k;
  
  k=0;
  if (key[KEY_F12]&&(key_shifts&KB_CTRL_FLAG)) Reset6502(R);
  if (key[KEY_F4]&&(key_shifts&KB_ALT_FLAG)) return INT_QUIT;
  if (key[KEY_UP]) k=11;
  if (key[KEY_DOWN]) k=10;
  if (key[KEY_LEFT]) k=8;
  if (key[KEY_RIGHT]) k=21;
  if (key[KEY_F2])
  {
   memset(mem,0,49152);
   Reset6502(&CPU);
   return INT_NONE;
  }
  if (key[KEY_F3])
  {
   int ok;

   ok=file_select_ex("Select Disk for Internal Drive",DrvSt[0].DiskFN,"DSK;DO",1024,0,0);
   if (ok)
   {
    UnmountDisk(0);
    DrvSt[0].DiskType = UnknownType;
    MountDisk(0);
   }
  }
  
  if (k) readkey(); else k=readkey()&0x7F;
  
  if ((k>='A')&&(k<='Z')) k+=('a'-'A');
   else if ((k>='a')&&(k<='z')) k-=('a'-'A');
  if (k) 
  {
   lastkey=k|0x80;
  }
 }

 return INT_NONE;
}

word XRun6502(M6502 *R)
{
  register pair J,K;
  register byte I;

  for(;;)
  {
#ifdef DEBUG
    /* Turn tracing on when reached trap address */
    if(R->PC.W==R->Trap) R->Trace=1;
    /* Call single-step debugger, exit if requested */
    if(R->Trace)
      if(!Debug6502(R)) return(R->PC.W);
#endif

    ctdif=0;

    ctdif=R->ICount;
    Exec6502(R);
    ClockTick+=(ctdif-R->ICount);

    /* If cycle counter expired... */
    if(R->ICount<=0)
    {
      /* If we have come after CLI, get INT_? from IRequest */
      /* Otherwise, get it from the loop handler            */
      if(R->AfterCLI)
      {
        I=R->IRequest;            /* Get pending interrupt     */
        R->ICount+=R->IBackup-1;  /* Restore the ICount        */
        R->AfterCLI=0;            /* Done with AfterCLI state  */
      }
      else
      {
        I=Loop6502(R);            /* Call the periodic handler */
        R->ICount=R->IPeriod;     /* Reset the cycle counter   */
      }

      if(I==INT_QUIT) return(R->PC.W); /* Exit if INT_QUIT     */
      if(I) Int6502(R,I);              /* Interrupt if needed  */
    }
  }

  /* Execution stopped */
  return(R->PC.W);
}

int main (int argc, char **argv)
{
 FILE *file;
 word ladd, llen;
 int badfile;
 struct {
  byte magic[8];
  byte zero[2];
  byte lodadr[2];
  byte len[2];
 } sosinterp_header;

 allegro_init();
 LOCK_FUNCTION(die);
 set_close_button_callback(die);
 install_keyboard();

 (mem[0xFFD0])=0;
 hideio=0;

 file=fopen("staremu.fnt", "rb");
 if (!file)
 {
  hasstarfont=0;
 }
 else
 {
  hasstarfont=1;
  fread(starfont,1,1024,file);
  fclose(file);
 }
 file=fopen("system.fnt", "rb");
 if (!file)
 {
  hassysfont=0;
 }
 else
 {
  hassysfont=1;
  fread(sysfont,1,1024,file);
  fclose(file);
 }
 if ((!hasstarfont)&&(!hassysfont))
 {
  fprintf (stderr, "Neither staremu.fnt nor system.fnt could be read.\n");
  return;
 }

 real3=1;
 file=fopen("apple3.rom", "rb");
 if (!file)
 {
  file=fopen("SOS.INTERP#090000", "rb");
  if (!file)
  {
   fprintf (stderr, "In order to function, Daplus 3 requires either an Apple /// firmware or\n"
                    "the file SOS.INTERP extracted from the Star Emulator diskette.\n\n"
                    "When extracted with CiderPress, the Star Emulator filename is\n"
                    "\"SOS.INTERP#090000\".  The Apple /// ROM filename is \"apple3.rom\".\n");
   return -1;
  }
 
  fread(&sosinterp_header,1,14,file);
  badfile=0;
  if (sosinterp_header.magic[0]!='S') badfile=1;
  if (sosinterp_header.magic[1]!='O') badfile=1;
  if (sosinterp_header.magic[2]!='S') badfile=1;
  if (sosinterp_header.magic[3]!=' ') badfile=1;
  if (sosinterp_header.magic[4]!='N') badfile=1;
  if (sosinterp_header.magic[5]!='T') badfile=1;
  if (sosinterp_header.magic[6]!='R') badfile=1;
  if (sosinterp_header.magic[7]!='P') badfile=1;
  if (sosinterp_header.zero[0]) badfile=1;
  if (sosinterp_header.zero[1]) badfile=1;
  ladd=sosinterp_header.lodadr[1];
  ladd<<=8;
  ladd+=sosinterp_header.lodadr[0];
  llen=sosinterp_header.len[1];
  llen<<=8;
  llen+=sosinterp_header.len[0];
  if (badfile)
  {
   fprintf (stderr, "The SOS.INTERP does not look valid.\n");
   fclose(file);
   return;
  }
  fread(&(mem[ladd]),1,llen,file);
  fclose(file);
  mem[0xFFFC]=sosinterp_header.lodadr[0];
  mem[0xFFFD]=sosinterp_header.lodadr[1];
  real3=0;
 }
 else
 {
  isrom=1;
  
  fread(rom,1,4096,file);
  fclose(file);
 }

 printf (DAPLUS_VERSION "\n"
         "Copyright " DAPLUS_DATE "Steve Nickolas\n"
         "Based on concepts and code from Dapple 1.25 copyright 2001-2003 Steve Nickolas\n"
         "CPU code copyright 1996 Marat Fayzullin and Alex Krasivsky\n");

 if (real3)
  printf ("Using an Apple /// firmware\n");
 else
  printf ("Star Emulator memory image loaded at $%04X.%04X\n", ladd, (ladd+llen)-1);

 InitDisk(6);

 Reset6502(&CPU);
 printf ("Emulation commencing at ip=$%04X...\n", CPU.PC.W);
 cpuspeed=1;
 CPU.IPeriod=65;

 corpse=0;

 set_color_depth(24);
 if (set_gfx_mode(GFX_AUTODETECT_WINDOWED,640,480,0,0))
 {
  printf("No 640x480x24 mode");
  goto death;
 }

 flushscreen(); /* init tty for fast operation */
 
 XRun6502(&CPU);

death:
 allegro_exit();
}
END_OF_MAIN()
